import logging
import os
import re
from glob import glob

from setuptools import Command
from setuptools._distutils.util import change_root, convert_path
from setuptools.command.build import _build


def parse_comma_separated_list(lang: str) -> list[str]:
    """Convert Comma-separated list to python tuple."""
    return [i.strip() for i in lang.split(",") if i.strip()]


def lang_from_dir(source_dir: os.PathLike) -> list[str]:
    """Get list of the languages that are stored in the source_dir."""
    re_po = re.compile(r"^([a-zA-Z_]+)\.po$")
    lang = []
    for i in os.listdir(source_dir):
        mo = re_po.match(i)
        if mo:
            lang.append(mo.group(1))
    return lang


def has_po(_c) -> bool:
    """Check if there is some not compiled files(.po) in source_dir."""
    return True


msg_params = "--check-format --check-domain -o"
SOURCE_DIR = "calculate/locale"


class build_mo(Command):
    """Subcommand of build command: build_mo."""

    description = "compile po files to mo files"
    user_options = [
        ("output-base=", "o", "mo-files base name"),
        ("source-dir=", None, "Directory with sources po files"),
        ("force", "f", "Force creation of mo files"),
        ("lang=", None, "Comma-separated list of languages " "to process"),
    ]

    boolean_options = ["force"]

    build_lib: str = None

    editable_mode: str = False

    def initialize_options(self) -> None:
        self.source_dir: str = None
        self.outfiles: list[str] = []
        self.lang = None

    def finalize_options(self) -> None:
        self.set_undefined_options("build")
        if self.source_dir is None:
            self.source_dir = SOURCE_DIR
        if self.lang is None:
            self.lang = lang_from_dir(self.source_dir)
        else:
            self.lang = parse_comma_separated_list(self.lang)

    def run(self):
        for locale in self.lang:
            localepath = os.path.join(self.source_dir, locale, "LC_MESSAGES")
            po_path = f"{self.source_dir}/{locale}"

            self.mkpath(po_path)
            self.mkpath(localepath)

            for po in glob(f"{po_path}/*.po"):
                if "packages.po" in po or "new.po" in po:
                    continue

                addon = f"{po_path}/packages.po"
                mo = os.path.basename(po)[:-2] + "mo"
                if "cl_update" in po and os.path.exists(addon):
                    cmd = f"msgfmt {msg_params} {localepath}/{mo} {po} {addon}"
                else:
                    cmd = f"msgfmt {msg_params} {localepath}/{mo} {po}"
                logging.info(cmd)
                os.system(cmd)

    def get_output_mapping(self) -> dict[str, str]:
        pass

    def get_outputs(self) -> list[str]:
        return self.outfiles

    def get_source_files(self) -> list[str]:
        pass


class build(_build):
    def run(self):
        self.run_command("build_mo")
        _build.run(self)


class install_data(Command):
    """Subcommand of install command: install_data."""

    description = "install data files"

    user_options = [
        (
            "install-dir=",
            "d",
            "base directory for installing data files " "[default: installation base dir]",
        ),
        ("root=", None, "install everything relative to this alternate root directory"),
        ("force", "f", "force installation (overwrite existing files)"),
        ("uses=", None, "Comma-separated list of USE flags to process"),
    ]

    boolean_options = ["force"]

    def run(self):
        self.mkpath(self.install_dir)
        for f in self.data_files:
            if isinstance(f, str):
                # it's a simple file, so copy it
                f = convert_path(f)
                if self.warn_dir:
                    self.warn(
                        "setup script did not provide a directory for "
                        f"'{f}' -- installing right in '{self.install_dir}'"
                    )
                (out, _) = self.copy_file(f, self.install_dir)
                self.outfiles.append(out)
            else:
                # it's a tuple with path to install to and a list of files
                if len(f) > 2 and self.uses is not None:
                    if f[2] not in self.uses:
                        continue

                dir = convert_path(f[0])
                if not os.path.isabs(dir):
                    dir = os.path.join(self.install_dir, dir)
                elif self.root:
                    dir = change_root(self.root, dir)
                self.mkpath(dir)

                if f[1] == []:
                    # If there are no files listed, the user must be
                    # trying to create an empty directory, so add the
                    # directory to the list of output files.
                    self.outfiles.append(dir)
                else:
                    # Copy files, adding them to the list of output files.
                    for data in f[1]:
                        # is's a simple filename without chmod
                        if isinstance(data, str):
                            chmod = None
                        else:
                            data, chmod = data
                        data = convert_path(data)
                        (out, _) = self.copy_file(data, dir)
                        old_mod = os.stat(out).st_mode & 0o777
                        if chmod and old_mod != chmod:
                            logging.info(f"changing mode of {data} from {old_mod:o} to {chmod:o}")
                            os.chmod(out, chmod)
                        self.outfiles.append(out)

    def initialize_options(self):
        self.install_dir = None
        self.uses = None
        self.outfiles = []
        self.root = None
        self.force = False
        self.data_files = self.distribution.data_files
        self.warn_dir = True

    def finalize_options(self):
        self.set_undefined_options(
            "install",
            ("install_data", "install_dir"),
            ("root", "root"),
            ("force", "force"),
        )
        if self.uses is not None:
            self.uses = parse_comma_separated_list(self.uses)
        else:
            self.uses = []

    def get_inputs(self):
        return self.data_files or []

    def get_outputs(self):
        return self.outfiles
